-------------------------------------------------------------------------------
-- Training for SQL recursive queries

-- Give the (EmpId, SuperId) pairs for all direct or indirect supervisors of 
-- employees 

/*
WITH RECURSIVE Supervision(EmpId, SuperId) AS (
  SELECT e.EmpId, e.SuperId
  FROM Employee e
  UNION
  SELECT s.EmpId, e.SuperId
  FROM Employee e, Supervision s
  WHERE s.SuperId = e.EmpId AND e.SuperId IS NOT NULL )
SELECT * 
FROM Supervision
ORDER BY EmpId, SuperId;

   empid   |  superid
-----------+-----------
 e1        |
 e2        | e1
 e3        | e1
 e4        | e1
 e4        | e2
 e5        | e1
 e5        | e2
 e6        | e1
 e6        | e2
 e7        | e1
 e7        | e3
 e8        | e1
 e8        | e3
 e9        | e1
 e9        | e3
 e9        | e8
*/

-- Give the (EmpId, EmpName, SuperId, SuperName) pairs for all 
-- direct or indirect supervisors of employees 

/*
WITH RECURSIVE Supervision(EmpId, EmpName, SuperId, SuperName) AS (
  SELECT e.EmpId, e.FName || ' ' || e.LName AS EmpName, 
    s.EmpId, s.FName || ' ' || s.LName AS SuperName
  FROM Employee e LEFT OUTER JOIN Employee s ON e.SuperId = s.EmpId
  UNION
  SELECT s.EmpId, s.EmpName, e2.EmpId, e2.FName || ' ' || e2.LName AS SuperName
  FROM Employee e1, Employee e2, Supervision s
  WHERE s.SuperId = e1.EmpId AND e1.SuperId = e2.EmpId )
SELECT * 
FROM Supervision
ORDER BY EmpId, SuperId;

   empid   |     empname      |  superid  |    supername
-----------+------------------+-----------+------------------
 e1        | James Borg       |           |
 e2        | Franklin Wong    | e1        | James Borg
 e3        | Jennifer Wallace | e1        | James Borg
 e4        | Ramesh Narayan   | e1        | James Borg
 e4        | Ramesh Narayan   | e2        | Franklin Wong
 e5        | John Smith       | e1        | James Borg
 e5        | John Smith       | e2        | Franklin Wong
 e6        | Joyce English    | e1        | James Borg
 e6        | Joyce English    | e2        | Franklin Wong
 e7        | Ahmad Jabbar     | e1        | James Borg
 e7        | Ahmad Jabbar     | e3        | Jennifer Wallace
 e8        | Alicia Zelaya    | e1        | James Borg
 e8        | Alicia Zelaya    | e3        | Jennifer Wallace
 e9        | Mary Stuart      | e1        | James Borg
 e9        | Mary Stuart      | e3        | Jennifer Wallace
 e9        | Mary Stuart      | e8        | Alicia Zelaya
*/

-- Give the (EmpId, EmpName, HierLevel, SuperId, SuperName) pairs for all   
-- director indirect supervisors of employees where HierLevel is 0 for the CEO,
-- 1 for the direct subordinates of the CEO, etc.

/*
WITH RECURSIVE Supervision(EmpId, EmpName, HierLevel, SuperId, SuperName) AS (
  SELECT e.EmpId, e.FName || ' ' || e.LName AS EmpName,
    CASE WHEN e.SuperId IS NULL THEN 0 ELSE 1 END,
    s.EmpId, s.FName || ' ' || s.LName AS SuperName
  FROM Employee e LEFT OUTER JOIN Employee s ON e.SuperId = s.EmpId
  UNION
  SELECT s.EmpId, s.EmpName, HierLevel + 1, e2.EmpId, e2.FName || ' ' || e2.LName AS SuperName
  FROM Employee e1, Employee e2, Supervision s
  WHERE s.SuperId = e1.EmpId AND e1.SuperId = e2.EmpId )
SELECT s1.EmpId, s1.EmpName, MaxHierLevel AS HierLevel, s1.SuperId, s1.SuperName 
FROM Supervision s1 CROSS JOIN LATERAL (
  SELECT MAX(HierLevel) AS MaxHierLevel 
  FROM Supervision s2 WHERE s1.EmpId = s2.EmpId )
ORDER BY EmpId, SuperId;

   empid   |     empname      | hierlevel |  superid  |    supername
-----------+------------------+-----------+-----------+------------------
 e1        | James Borg       |         0 |           |
 e2        | Franklin Wong    |         1 | e1        | James Borg
 e3        | Jennifer Wallace |         1 | e1        | James Borg
 e4        | Ramesh Narayan   |         2 | e1        | James Borg
 e4        | Ramesh Narayan   |         2 | e2        | Franklin Wong
 e5        | John Smith       |         2 | e1        | James Borg
 e5        | John Smith       |         2 | e2        | Franklin Wong
 e6        | Joyce English    |         2 | e1        | James Borg
 e6        | Joyce English    |         2 | e2        | Franklin Wong
 e7        | Ahmad Jabbar     |         2 | e1        | James Borg
 e7        | Ahmad Jabbar     |         2 | e3        | Jennifer Wallace
 e8        | Alicia Zelaya    |         2 | e1        | James Borg
 e8        | Alicia Zelaya    |         2 | e3        | Jennifer Wallace
 e9        | Mary Stuart      |         3 | e1        | James Borg
 e9        | Mary Stuart      |         3 | e3        | Jennifer Wallace
 e9        | Mary Stuart      |         3 | e8        | Alicia Zelaya
*/

-- Give the for each EmpId the array of ids for its direct or indirect 
-- subordinates

/*
WITH RECURSIVE Supervision(EmpId, SubordId) AS (
  SELECT e.SuperId, e.EmpId
  FROM Employee e
  WHERE e.SuperId IS NOT NULL
  UNION
  SELECT s.EmpId, e.EmpId
  FROM Supervision s, Employee e
  WHERE s.SubordId = e.SuperId AND e.SuperId IS NOT NULL )
SELECT EmpId, array_agg(SubordId ORDER BY SubordId) AS Subords
FROM Supervision
GROUP BY EmpId
ORDER BY EmpId;

 empid |            subords
-------+-------------------------------
 e1    | {e10,e2,e3,e4,e5,e6,e7,e8,e9}
 e2    | {e4,e5,e6}
 e3    | {e10,e7,e8,e9}
 e8    | {e10,e9}
 e9    | {e10}
(5 rows)

WITH RECURSIVE Supervision(EmpId, SuperId) AS (
  SELECT e.EmpId, e.SuperId 
  FROM Employee e
  WHERE e.SuperId IS NOT NULL
  UNION
  SELECT s.EmpId, e.SuperId
  FROM Supervision s, Employee e
  WHERE s.SuperId = e.EmpId AND e.SuperId IS NOT NULL )
SELECT EmpId, array_agg(SuperId ORDER BY SuperId) AS Subords
FROM Supervision
GROUP BY EmpId
ORDER BY EmpId;

 empid |  subords
-------+------------
 e2    | {e1}
 e3    | {e1}
 e4    | {e1,e2}
 e5    | {e1,e2}
 e6    | {e1,e2}
 e7    | {e1,e3}
 e8    | {e1,e3}
 e9    | {e1,e3,e8}
(8 rows)
*/

-------------------------------------------------------------------------------
-- 1) The age of employees must be greater than 18.

-- Using a CHECK constraint

-- ALTER TABLE Employee
-- ADD CONSTRAINT employee_Age18
-- CHECK ( BDate + interval '18 years' <= CURRENT_DATE );

-- ALTER TABLE Employee
-- DROP CONSTRAINT employee_Age18;

-- Using a trigger

CREATE OR REPLACE FUNCTION age18()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF NEW.BDate + interval '18 years' > CURRENT_DATE THEN
    RAISE EXCEPTION 
      'The age of an employee must be greater than 18'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
    END IF;
  RETURN NEW;
END; $$;

CREATE OR REPLACE TRIGGER age18
BEFORE INSERT OR UPDATE OF BDate ON Employee
FOR EACH ROW
EXECUTE PROCEDURE age18();

/* 
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-2017',
'731 Fondren, Houston, TX', 'M', 30000, 'e9', 3, '01-01-1985');
-- ERROR:  The age of an employee must be greater than 18
-- CONTEXT:  PL/pgSQL function age18() line 5 at RAISE
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1965',
'731 Fondren, Houston, TX', 'M', 30000, 'e9', 3, '01-01-1985');
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';

UPDATE Employee SET BDate = '09-05-2017' WHERE EmpId = 'e9';
-- ERROR:  Constraint Violation: The age of an employee must be greater than 18
-- CONTEXT:  PL/pgSQL function age18() line 5 at RAISE
UPDATE Employee SET BDate = '10-11-1960' WHERE EmpId = 'e9';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET BDate = '10-11-1959' WHERE EmpId = 'e9';
*/

-------------------------------------------------------------------------------
-- 2) The supervisor of an employee must be older than the employee

CREATE OR REPLACE FUNCTION supervisorAge()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
  rowcnt integer;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  SELECT COUNT(*) INTO rowcnt
  FROM Employee e
  WHERE ( NEW.SuperId = e.EmpId AND NEW.BDate < e.BDate ) OR
        ( e.SuperId = NEW.EmpId AND e.BDate < NEW.BDate );
  IF rowcnt > 0 THEN
    RAISE EXCEPTION 
      'The age of an employee must be less than the age of his/her supervisor'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NEW;
END; $$;

CREATE OR REPLACE TRIGGER supervisorAge
BEFORE INSERT OR UPDATE OF BDate, SuperId ON Employee
FOR EACH ROW
EXECUTE PROCEDURE supervisorAge();

/*
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1949',
'450 Stone, Houston, TX', 'F', 55000, 'e8', 1, '01-01-1980');
-- ERROR:  The age of an employee must be less than the age of his/her supervisor
-- CONTEXT:  PL/pgSQL function supervisorage() line 9 at RAISE
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
'450 Stone, Houston, TX', 'F', 55000, 'e8', 1, '01-01-1980');
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';

UPDATE Employee SET BDate = '19-07-2000' WHERE EmpId = 'e8';
-- ERROR:  The age of an employee must be less than the age of his/her supervisor
-- CONTEXT:  PL/pgSQL function supervisorage() line 9 at RAISE
UPDATE Employee SET BDate = '19-07-1959' WHERE EmpId = 'e8';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET BDate = '19-07-1958' WHERE EmpId = 'e8';
-- UPDATE 1
*/

-------------------------------------------------------------------------------
-- 4) The salary of an employee cannot be greater than the salary of his/her supervisor.”

CREATE OR REPLACE FUNCTION supervisorSalary()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
  rowcnt integer;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  SELECT COUNT(*) INTO rowcnt
  FROM Employee e
  WHERE ( NEW.SuperId = e.EmpId AND NEW.Salary > e.Salary ) OR
        ( e.SuperId = NEW.EmpId AND e.Salary > NEW.Salary );
  IF rowcnt > 0 THEN
    RAISE EXCEPTION 
      'The salary of an employee must be less than the salary of his/her supervisor'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NEW;
END; $$;

CREATE OR REPLACE TRIGGER supervisorSalary
BEFORE INSERT OR UPDATE OF Salary, SuperId ON Employee
FOR EACH ROW
EXECUTE PROCEDURE supervisorSalary();

/*
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
'450 Stone, Houston, TX', 'F', 55000, 'e8', 1, '01-01-1980');
-- ERROR:  The salary of an employee must be less than the salary of his/her supervisor
-- CONTEXT:  PL/pgSQL function supervisorsalary() line 13 at RAISE
INSERT INTO Employee VALUES
('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
'450 Stone, Houston, TX', 'F', 23000, 'e8', 1, '01-01-1980');
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';

UPDATE Employee SET Salary = 50000 WHERE EmpId = 'e8';
-- ERROR:  The age of an employee must be less than the age of his/her supervisor
-- CONTEXT:  PL/pgSQL function supervisorage() line 9 at RAISE
UPDATE Employee SET Salary = 26000 WHERE EmpId = 'e8';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET Salary = 25000 WHERE EmpId = 'e8';
*/

-------------------------------------------------------------------------------
-- 4) The manager of a department must be an employee of that department.

ALTER TABLE Employee
-- DROP CONSTRAINT IF EXISTS UN_Employee_EmpId_DNo,
ADD CONSTRAINT UN_Employee_EmpId_DNo
UNIQUE(EmpId, DNumber);

ALTER TABLE Department
-- DROP CONSTRAINT IF EXISTS FK_Employee_EmpId_DNo,
ADD CONSTRAINT FK_Employee_EmpId_DNo
FOREIGN KEY(MgrId, DNumber)
REFERENCES Employee(EmpId, DNumber);

/*
INSERT INTO Department VALUES
  (9, 'Sales', 'e1', CURRENT_DATE, 1);
-- ERROR:  insert or update on table "department" violates foreign key constraint "fk_employee_empid_dno"
-- DETAIL:  Key (mgrid, dnumber)=(e1, 9) is not present in table "employee".
  
BEGIN TRANSACTION;
  ALTER TABLE Employee DISABLE TRIGGER ALL;
  ALTER TABLE Department DISABLE TRIGGER ALL;
  INSERT INTO Employee VALUES
    ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e8', 9, '01-01-1980');
  INSERT INTO Department VALUES
    (9, 'Sales', 'e10', CURRENT_DATE, 1);
  ALTER TABLE Department ENABLE TRIGGER ALL;
  ALTER TABLE Employee ENABLE TRIGGER ALL;
END TRANSACTION;
-- COMMIT
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';
DELETE FROM Department WHERE DNumber = 9;

UPDATE Department SET MgrId = 'e2' WHERE DNumber = 3;
-- ERROR:  insert or update on table "department" violates foreign key constraint "fk_employee_empid_dno"
-- DETAIL:  Key (mgrid, dnumber)=(e2, 3) is not present in table "employee".
UPDATE Department SET MgrId = 'e4' WHERE DNumber = 3;
-- UPDATE 1
-- Revert back to the original database state
UPDATE Department SET MgrId = 'e3' WHERE DNumber = 3;

UPDATE Employee SET DNumber = 4 WHERE EmpId = 'e2';
-- ERROR:  update or delete on table "employee" violates foreign key constraint "fk_employee_empid_dno" on table "department"
-- DETAIL:  Key (empid, dnumber)=(e2, 2) is still referenced from table "department".
UPDATE Employee SET DNumber = 2 WHERE EmpId = 'e5';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET DNumber = 3 WHERE EmpId = 'e5';
*/

-------------------------------------------------------------------------------
-- 5) The location of a project must be one of the locations of its department.

ALTER TABLE Project
ADD CONSTRAINT FK_Project_DeptLocations
FOREIGN KEY(DNumber, PLocation)
REFERENCES DeptLocations(DNumber, DLocation);

/*
INSERT INTO Project VALUES
  (40, 'Dashboard', 'Stafford', 3);
-- ERROR:  insert or update on table "project" violates foreign key constraint "fk_project_deptlocations"
-- DETAIL:  Key (dnumber, plocation)=(3, Stafford) is not present in table "deptlocations".
INSERT INTO Project VALUES
  (40, 'Dashboard', 'Houston', 3);
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Project WHERE PNumber = 40;

UPDATE Project SET PLocation = 'Stafford' WHERE PNumber = 1;
-- ERROR:  insert or update on table "project" violates foreign key constraint "fk_project_deptlocations"
-- DETAIL:  Key (dnumber, plocation)=(3, Stafford) is not present in table "deptlocations".
UPDATE Project SET PLocation = 'Houston' WHERE PNumber = 1;
-- UPDATE 1
-- Revert back to the original database state
UPDATE Project SET PLocation = 'Bellaire' WHERE PNumber = 1;
-- UPDATE 1
*/

-------------------------------------------------------------------------------
-- 6) The hire date of employees must be greater than their birth date.

ALTER TABLE Employee
-- DROP CONSTRAINT IF EXISTS HireDate_BDate,
ADD CONSTRAINT HireDate_BDate
CHECK( HireDate > BDate );

/*
INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1981',
  '450 Stone, Houston, TX', 'F', 23000, 'e8', 3, '01-01-1980');
-- ERROR:  new row for relation "employee" violates check constraint "hiredate_bdate"
-- DETAIL:  Failing row contains (Elizabeth, A, Tudor, e10, 1981-11-10, 450 Stone, Houston, TX, M, 23000.00, e8, 5, 1980-01-01).
INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1979',
  '450 Stone, Houston, TX', 'F', 23000, 'e8', 3, '01-01-1980');
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';
-- DELETE 1

UPDATE Employee SET Bdate = '01-01-1991' WHERE EmpId = 'e9';
-- ERROR:  new row for relation "employee" violates check constraint "hiredate_bdate"
-- DETAIL:  Failing row contains (e9, Mary, A, Stuart, 1991-01-01, 450 Stone, Houston, TX, M, 24000.00, e8, 1, 1990-01-01).
UPDATE Employee SET Bdate = '01-01-1979' WHERE EmpId = 'e9';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET Bdate = '10-11-1959' WHERE EmpId = 'e9';
-- UPDATE 1
*/

-------------------------------------------------------------------------------
-- 7) A supervisor must be hired at least 1 year before all her subordinates.

CREATE OR REPLACE FUNCTION hireSuperv()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
  rowcnt integer;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  SELECT COUNT(*) INTO rowcnt
  FROM Employee e
  WHERE ( NEW.SuperId = e.EmpId AND NEW.HireDate - interval '1 year' < e.HireDate ) OR
        ( e.SuperId = NEW.EmpId AND NEW.HireDate + interval '1 year' < e.HireDate );
  IF rowcnt > 0 THEN
    RAISE EXCEPTION 
      'A supervisor must be hired at least 1 year before all her subordinates'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NEW;
END; $$;

CREATE OR REPLACE TRIGGER hireSuperv
BEFORE INSERT OR UPDATE OF HireDate, SuperId ON Employee
FOR EACH ROW
EXECUTE PROCEDURE hireSuperv();

/*
INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
  '450 Stone, Houston, TX', 'F', 23000, 'e8', 3, '01-01-1980');
-- ERROR:  A supervisor must be hired at least 1 year before all her subordinates 1980-01-01).
INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
  '450 Stone, Houston, TX', 'F', 23000, 'e8', 3, '01-01-1986');
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';
-- DELETE 1

UPDATE Employee SET HireDate = '01-01-1979' WHERE EmpId = 'e2';
-- ERROR:  A supervisor must be hired at least 1 year before all her subordinates
-- CONTEXT:  PL/pgSQL function hiresuperv() line 13 at RAISE
UPDATE Employee SET HireDate = '01-01-1984' WHERE EmpId = 'e2';
-- UPDATE 1
-- Revert back to the original database state
UPDATE Employee SET HireDate = '01-01-1982' WHERE EmpId = 'e2';
-- UPDATE 1
*/
  
-------------------------------------------------------------------------------
-- 8) The attribute Department.NoEmployees is a derived attribute from Employee.DNumber

/*
-- Recomputing derived attribute from scratch ROW TRIGGER
CREATE OR REPLACE FUNCTION Dept_NoEmp_derived()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  UPDATE Department d
  SET NoEmployees = (
    SELECT COUNT(*)
    FROM Employee e
    WHERE e.DNumber = d.DNumber )
  WHERE d.DNumber = NEW.DNumber OR d.DNumber = OLD.DNumber;
  RETURN NEW;
END; $$;
*/

/*
-- Recomputing derived attribute incrementally ROW TRIGGER
CREATE OR REPLACE FUNCTION Dept_NoEmp_derived()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  UPDATE Department d
  SET NoEmployees = NoEmployees - 1
  WHERE OLD.DNumber = d.DNumber;
  UPDATE Department d
  SET NoEmployees = NoEmployees + 1
  WHERE NEW.DNumber = d.DNumber;
  RETURN NEW;
END; $$;

CREATE OR REPLACE TRIGGER Dept_NoEmp_derived
AFTER INSERT OR UPDATE OF DNumber OR DELETE ON Employee
FOR EACH ROW
EXECUTE PROCEDURE Dept_NoEmp_derived();
*/

-- Recomputing derived attribute incrementally
CREATE OR REPLACE FUNCTION Dept_NoEmp_derived()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF (TG_OP = 'INSERT') THEN
    UPDATE Department u
    SET NoEmployees = NoEmployees +
      ( SELECT COUNT(*) FROM Inserted i WHERE u.DNumber = i.DNumber )
    WHERE u.DNumber IN ( SELECT DNumber FROM Inserted );
  ELSIF (TG_OP = 'UPDATE') THEN
    UPDATE Department u
    SET NoEmployees = NoEmployees +
      ( SELECT COUNT(*) FROM Inserted i WHERE u.DNumber = i.DNumber ) -
      ( SELECT COUNT(*) FROM Deleted d WHERE u.DNumber = d.DNumber )
    WHERE u.DNumber IN ( SELECT DNumber FROM Inserted ) OR
      DNumber IN ( SELECT DNumber FROM Deleted );
  ELSIF (TG_OP = 'DELETE') THEN
    UPDATE Department u
    SET NoEmployees = NoEmployees -
      ( SELECT COUNT(*) FROM Deleted d WHERE u.DNumber = d.DNumber )
    WHERE u.DNumber IN ( SELECT DNumber FROM Deleted );
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Dept_NoEmp_derived_ins
AFTER INSERT ON Employee
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Dept_NoEmp_derived();

CREATE OR REPLACE TRIGGER Dept_NoEmp_derived_upd
AFTER UPDATE ON Employee
REFERENCING OLD TABLE AS Deleted NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Dept_NoEmp_derived();

CREATE OR REPLACE TRIGGER Dept_NoEmp_derived_del
AFTER DELETE ON Employee
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Dept_NoEmp_derived();

/*
SELECT DNumber, COUNT(*) FROM Employee GROUP BY Dnumber ORDER BY 1;
 dnumber | count
---------+-------
       1 |     2
       2 |     3
       3 |     4

SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | count
---------+-------
       1 |     2
       2 |     3
       3 |     4

INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
  '450 Stone, Houston, TX', 'F', 23000, 'e8', 3, '01-01-1986');
  
SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | noemployees
---------+-------------
       1 |           2
       2 |           3
       3 |           5

UPDATE Employee SET DNumber = 2 WHERE EmpId = 'e10';

SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | noemployees
---------+-------------
       1 |           2
       2 |           4
       3 |           4

-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';
-- DELETE 1

SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | noemployees
---------+-------------
       1 |           2
       2 |           3
       3 |           4
*/

CREATE OR REPLACE FUNCTION Dept_NoEmp_derived_dept()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
    SELECT *
    FROM Inserted i
    WHERE NoEmployees <> ( 
      SELECT COUNT(*)
      FROM Employee e
      WHERE e.DNumber = i. DNumber ) ) THEN
    RAISE EXCEPTION 
      'Department.NoEmployees is a derived attribute from Employee.DNo'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Dept_NoEmp_derived_dept_ins
AFTER INSERT OR UPDATE ON Department
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Dept_NoEmp_derived_dept();

/*
BEGIN TRANSACTION;
  ALTER TABLE Employee DISABLE TRIGGER ALL;
  ALTER TABLE Department DISABLE TRIGGER ALL;
  INSERT INTO Employee VALUES
    ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e8', 9, '01-01-1980');
  INSERT INTO Department VALUES
    ( 9, 'Sales','e10', CURRENT_DATE, 1);
  ALTER TABLE Department ENABLE TRIGGER ALL;
  ALTER TABLE Employee ENABLE TRIGGER ALL;
END TRANSACTION;

SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | noemployees
---------+-------------
       1 |           2
       2 |           3
       3 |           4
       9 |           1

-- Revert back to the original database state
DELETE FROM Employee WHERE EmpId = 'e10';
-- DELETE 1
DELETE FROM Department WHERE Dnumber = 9;
-- DELETE 1

SELECT DNumber, NoEmployees FROM Department ORDER BY 1;
 dnumber | noemployees
---------+-------------
       1 |           2
       2 |           3
       3 |           4
*/

-------------------------------------------------------------------------------
-- 9) An employee works at most in 4 projects

CREATE OR REPLACE FUNCTION Emp_Max_Proj()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM WorksOn w
      GROUP BY w.EmpId
      HAVING COUNT(*) > 4 ) THEN
    RAISE EXCEPTION 
      'An employee works at most in 4 projects'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Emp_Max_Proj
AFTER INSERT OR UPDATE ON WorksOn
FOR EACH STATEMENT
EXECUTE PROCEDURE Emp_Max_Proj();

/*
INSERT INTO WorksOn VALUES
('e3', 10, 32.5),
('e3', 20, 7.5);
-- ERROR:  An employee works at most in 4 projects

INSERT INTO WorksOn VALUES
('e3', 10, 32.5);
-- INSERT 0 1
-- Revert back to the original database state
DELETE FROM WorksOn WHERE EmpId = 'e3' AND PNumber = 10;
-- DELETE 1
*/

-------------------------------------------------------------------------------
-- 10) An employee works between 30 and 50 h/week on all its projects

CREATE OR REPLACE FUNCTION Emp_TotHours_Proj()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM WorksOn
      GROUP BY EmpId
      HAVING ( SUM(Hours) < 30 ) OR ( SUM(Hours) > 50 ) ) THEN
    RAISE EXCEPTION 
      'An employee works between 30 and 50 h/week on all its projects'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Emp_TotHours_Proj
AFTER INSERT OR UPDATE OR DELETE ON WorksOn
FOR EACH STATEMENT
EXECUTE PROCEDURE Emp_TotHours_Proj();

/*
INSERT INTO WorksOn VALUES ('e1', 20, 30.0);
-- ERROR:  An employee works at between 30 and 50 h/week on all its projects
INSERT INTO WorksOn VALUES ('e1', 20, 10.0);
-- INSERT 0 1

UPDATE WorksOn SET Hours = 30 WHERE EmpId = 'e1' AND PNumber = 20;
-- ERROR:  An employee works at between 30 and 50 h/week on all its projects
UPDATE WorksOn SET Hours = 15 WHERE EmpId = 'e1' AND PNumber = 20;
-- UPDATE 1
DELETE FROM WorksOn WHERE EmpId = 'e1' AND PNumber = 10;
-- ERROR:  An employee works at between 30 and 50 h/week on all its projects

-- Revert back to the original database state
DELETE FROM WorksOn WHERE EmpId = 'e1' AND PNumber = 20;
-- DELETE 1
*/

-------------------------------------------------------------------------------
-- 11) A project can have at most 2 employees working on the project 10 hours or less

CREATE OR REPLACE FUNCTION Proj_NoEmp_10Hours()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM WorksOn 
      WHERE Hours <= 10 
      GROUP BY PNumber
      HAVING COUNT(*) > 2 ) THEN
    RAISE EXCEPTION 
      'A project can have at most 2 employees working on the project less than 10 hours'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Proj_NoEmp_10Hours
AFTER INSERT OR UPDATE OR DELETE ON WorksOn
FOR EACH STATEMENT
EXECUTE PROCEDURE Proj_NoEmp_10Hours();

/*
INSERT INTO WorksOn VALUES ('e2', 1, 10.0), ('e4', 1, 10.0);
-- ERROR:  A project can have at most 2 employees working on the project less than 10 hours
INSERT INTO WorksOn VALUES ('e1', 1, 15.0), ('e4', 1, 10.0);
-- INSERT 0 2
UPDATE WorksOn SET Hours = 10 WHERE PNumber = 20 AND EmpId = 'e7';
-- ERROR:  A project can have at most 2 employees working on the project less than 10 hours
*/
UPDATE WorksOn SET Hours = 15 WHERE EmpId = 'e7' AND PNumber = 20;

-------------------------------------------------------------------------------
-- 12) Only department managers can work 5 hours or less on a project

CREATE OR REPLACE FUNCTION Mgrs_LT_5Hours_WorksOn()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM Inserted 
      WHERE Hours <= 5 AND EmpId NOT IN (
        SELECT MgrId FROM Department ) ) THEN
    RAISE EXCEPTION 
      'Only department managers can work 5 hours or less on a project'
       USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Mgrs_LT_5Hours_WorksOn_ins
AFTER INSERT ON WorksOn
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgrs_LT_5Hours_WorksOn();

CREATE OR REPLACE TRIGGER Mgrs_LT_5Hours_WorksOn_upd
AFTER UPDATE ON WorksOn
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgrs_LT_5Hours_WorksOn();

/*
INSERT INTO WorksOn VALUES ('e8', 10, 5.0);
-- ERROR:  Only department managers can work 5 hours or less on a project

UPDATE WorksOn SET Hours = 5 WHERE EmpId = 'e8' AND PNumber = 20;
-- ERROR:  Only department managers can work 5 hours or less on a project
UPDATE WorksOn SET Hours = 10 WHERE EmpId = 'e8' AND PNumber = 20;

*/

CREATE OR REPLACE FUNCTION Mgrs_LT_5Hours_Department()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM Deleted 
      WHERE MgrId NOT IN (SELECT MgrId FROM Department ) AND
            MgrId IN (SELECT EmpId FROM WorksOn WHERE Hours <= 5 ) ) THEN
    RAISE EXCEPTION 
      'Only department managers can work 5 hours or less on a project'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Mgrs_LT_5Hours_Department_ins
AFTER UPDATE ON Department
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgrs_LT_5Hours_Department();

CREATE OR REPLACE TRIGGER Mgrs_LT_5Hours_Department_upd
AFTER DELETE ON Department
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgrs_LT_5Hours_Department();

/*
BEGIN TRANSACTION;
  ALTER TABLE Employee DISABLE TRIGGER ALL;
  ALTER TABLE Department DISABLE TRIGGER ALL;
  INSERT INTO Employee VALUES
    ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e8', 9, '01-01-1980');
  INSERT INTO Department VALUES
    (9, 'Sales', 'e10', CURRENT_DATE, 1);
  ALTER TABLE Department ENABLE TRIGGER ALL;
  ALTER TABLE Employee ENABLE TRIGGER ALL;
  INSERT INTO WorksOn VALUES ('e10', 1, 5.0), ('e10', 2, 25.0);
END TRANSACTION;
-- COMMIT
UPDATE Department SET MgrId = 'e7' WHERE DNumber = 4;
-- ERROR:  Only department managers can work 5 hours or less on a project
*/

-------------------------------------------------------------------------------
-- 13) Employees who are not supervisors must work at least 10 hours on every project they work

CREATE OR REPLACE FUNCTION Superv_10Hours_WorksOn()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM Inserted 
      WHERE Hours < 10 AND EmpId NOT IN (
        SELECT SuperId
        FROM Employee
        WHERE SuperId IS NOT NULL ) ) THEN
    RAISE EXCEPTION 
      'Employees who are not supervisors must work at least 10 hours on every project they work'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Superv_10Hours_WorksOn_ins
AFTER INSERT ON WorksOn
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Superv_10Hours_WorksOn();

CREATE OR REPLACE TRIGGER Superv_10Hours_WorksOn_upd
AFTER UPDATE ON WorksOn
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Superv_10Hours_WorksOn();

/*
INSERT INTO WorksOn VALUES
  ('e9', 20, 30);
-- INSERT 0 1
INSERT INTO WorksOn VALUES
  ('e4', 20, 7.5);
-- ERROR: Employees who are not supervisors must work at least 10 hours on every project they work
INSERT INTO WorksOn VALUES
  ('e4', 20, 10);
-- INSERT 0 1
*/

CREATE OR REPLACE FUNCTION Superv_10Hours_Employee()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM Deleted 
      WHERE SuperId NOT IN (
        SELECT SuperId
        FROM Employee
        WHERE SuperId IS NOT NULL ) AND
        SuperId IN (
        SELECT EmpId
        FROM WorksOn
        WHERE Hours < 10 ) ) THEN
    RAISE EXCEPTION 
      'Employees who are not supervisors must work at least 10 hours on every project they work'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Superv_10Hours_Employee_ins
AFTER UPDATE ON Employee
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Superv_10Hours_Employee();

CREATE OR REPLACE TRIGGER Superv_10Hours_Employee_upd
AFTER DELETE ON Employee
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Superv_10Hours_Employee();

/*
INSERT INTO WorksOn VALUES
  ('e8', 20, 7.5);
-- INSERT 0 1
UPDATE Employee SET SuperId = 'e7' WHERE SuperId = 'e8';
-- ERROR:  Employees who are not supervisors must work at least 10 hours on every project they work
*/

-------------------------------------------------------------------------------
-- 14) The manager of a department must work at least 5 hours on all projects controlled by the department.

/*
CREATE OR REPLACE FUNCTION Mgr_5Hours_Department()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM ( Inserted i JOIN Project p ON i.DNumber = p.DNumber ) 
        LEFT OUTER JOIN WorksOn w ON MgrId = EmpId AND p.PNumber = w.PNumber
      WHERE Hours IS NULL OR Hours < 5 ) THEN
    RAISE EXCEPTION 
      'The manager of a department must work at least 5 hours on all projects controlled by the department'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Mgr_5Hours_Department_ins
AFTER INSERT ON Department
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_Department();

CREATE OR REPLACE TRIGGER Mgr_5Hours_Department_upd
AFTER UPDATE ON Department
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_Department();
*/
/*
INSERT INTO Project VALUES
  (69, 'Product69', 'Houston', 1);
*/

CREATE OR REPLACE FUNCTION Mgr_5Hours_Project()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF EXISTS ( 
      SELECT 1
      FROM ( Inserted i JOIN Department d ON d.DNumber = i.DNumber )
      LEFT OUTER JOIN WorksOn w ON MgrId = EmpId AND i.PNumber = w.PNumber
      WHERE ( Hours IS NULL OR Hours < 5 ) ) THEN
    RAISE EXCEPTION 
      'The manager of a department must work at least 5 hours on all projects controlled by the department'
      USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Mgr_5Hours_Project_ins
AFTER INSERT ON Project
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_Project();

CREATE OR REPLACE TRIGGER Mgr_5Hours_Project_upd
AFTER UPDATE ON Project
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_Project();

/*
INSERT INTO Project VALUES
  (69, 'Product69', 'Houston', 1);
-- ERROR: The manager of a department must work at least 5 hours on all projects controlled by the department
BEGIN TRANSACTION;
  ALTER TABLE Project DISABLE TRIGGER ALL;
  ALTER TABLE WorksOn DISABLE TRIGGER ALL;
  INSERT INTO Project VALUES
    (69, 'Product69', 'Houston', 1);
  INSERT INTO WorksOn VALUES
    ('e1', 69, 10);
  ALTER TABLE Project ENABLE TRIGGER ALL;
  ALTER TABLE WorksOn ENABLE TRIGGER ALL;
END TRANSACTION;
-- COMMIT

UPDATE Project VALUES
  (69, 'Product69', 'Houston', 1);
*/

CREATE OR REPLACE FUNCTION Mgr_5Hours_WorksOn()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  IF (TG_OP = 'UPDATE') THEN
    IF EXISTS ( 
      SELECT 1
      FROM ( Department d JOIN Project p on d.DNumber = p.DNumber)
      LEFT OUTER JOIN WorksOn w ON d.MgrId = w.EmpId AND p.PNumber = w.PNumber
      WHERE d.MgrId IN ( SELECT EmpId FROM Deleted ) AND 
        d.MgrId NOT IN ( SELECT EmpId FROM Inserted ) AND 
        ( w.Hours IS NULL OR w.Hours <= 5 ) ) THEN
      RAISE EXCEPTION 
        'The manager of a department must work at least 5 hours on all projects controlled by the department'
        USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
    END IF;
  END IF;
  IF (TG_OP = 'UPDATE') THEN
    IF EXISTS ( 
      SELECT 1
      FROM ( Department d JOIN Project p on d.DNumber = p.DNumber)
      LEFT OUTER JOIN WorksOn w ON d.MgrId = w.EmpId AND p.PNumber = w.PNumber
      WHERE d.MgrId IN ( SELECT EmpId FROM Deleted ) AND 
        ( w.Hours IS NULL OR w.Hours <= 5 ) ) THEN
      RAISE EXCEPTION 
        'The manager of a department must work at least 5 hours on all projects controlled by the department'
        USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
    END IF;
  END IF;
  RETURN NULL;
END; $$;

CREATE OR REPLACE TRIGGER Mgr_5Hours_WorksOn_upd
AFTER UPDATE ON WorksOn
REFERENCING OLD TABLE AS Deleted NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_WorksOn();

CREATE OR REPLACE TRIGGER Mgr_5Hours_WorksOn_del
AFTER DELETE ON WorksOn
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Mgr_5Hours_WorksOn();

/*
UPDATE WorksOn SET EmpId = 'e3' WHERE PNumber = 10;
-- 
DELETE FROM WorksOn WHERE EmpId = 'e1' AND PNumber = 10;

*/




/*
BEGIN TRANSACTION;
  ALTER TABLE Employee DISABLE TRIGGER ALL;
  ALTER TABLE Department DISABLE TRIGGER ALL;
  INSERT INTO Employee VALUES
    ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e8', 9, '01-01-1980');
  INSERT INTO Department VALUES
    (9, 'Sales', 'e10', CURRENT_DATE, 1);
  ALTER TABLE Department ENABLE TRIGGER ALL;
  ALTER TABLE Employee ENABLE TRIGGER ALL;
  INSERT INTO DeptLocations VALUES
    (9, 'Houston');
END TRANSACTION;

INSERT INTO Project VALUES
  (69, 'Product69', 'Houston', 9);
*/


-------------------------------------------------------------------------------
-- 15) The attribute Employee.SuperId is a derived attribute computed as follows. 
-- Department managers are supervised by the manager of Department 1 (Headquarters). 
-- Employees that are not managers are supervised by the manager of their department. 
-- Finally, the manager of Department 1 has a NULL value in attribute SuperId.

CREATE OR REPLACE FUNCTION Employee_SuperId_derived_Department()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  -- if UPDATE(MgrId) 
  UPDATE Employee e
  SET SuperId = (
    SELECT 
      CASE 
        WHEN e.EmpId != d.MgrId THEN d.MgrId
        WHEN e.EmpId = d.MgrId AND DNumber != 1 THEN 
          ( SELECT MgrId FROM Department WHERE DNumber = 1 )
        ELSE NULL
      END
    FROM Department d
    WHERE e.DNumber = d.DNumber )
  -- if the department manager changes all employees of the department
  -- must be UPDATEd
  WHERE ( e.DNumber IN (SELECT DNumber FROM Inserted ) )
  -- if the manager of department 1 changes, all department managers
  -- must be updated
  OR ( 1 IN ( SELECT DNumber FROM Inserted ) AND
       EmpId IN ( SELECT MgrId FROM Department ) );
END; $$;

CREATE OR REPLACE TRIGGER Employee_SuperId_derived_Department_ins
AFTER INSERT ON Department
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_derived_Department();

CREATE OR REPLACE TRIGGER Employee_SuperId_derived_Department_upd
AFTER UPDATE ON Department
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_derived_Department();

/*

UPDATE Department SET MgrId = 'e7' WHERE DNumber = 2;

*/

CREATE OR REPLACE FUNCTION Employee_SuperId_derived_Employee()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  -- if UPDATE(DNumber)
  UPDATE Employee e SET 
    SuperId = (
      SELECT 
        CASE 
          WHEN EmpId != MgrId THEN d.MgrId
          WHEN EmpId = MgrId AND i.DNumber != 1 THEN 
            ( SELECT MgrId FROM Department WHERE DNumber = 1 )
          ELSE NULL
        END
      FROM Inserted i, Department d
      WHERE e.EmpId = i.EmpId AND i.DNumber = d.DNumber )
  WHERE e.EmpId IN ( SELECT EmpId FROM Inserted );
END; $$;

CREATE OR REPLACE TRIGGER Employee_SuperId_derived_Employee_ins
AFTER INSERT ON Employee
REFERENCING NEW TABLE AS Inserted
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_derived_Employee();

CREATE OR REPLACE TRIGGER Mgr_5Hours_WorksOn_upd
AFTER UPDATE ON Employee
REFERENCING OLD TABLE AS Deleted
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_derived_Employee();

/*
SELECT EmpId, Dnumber, SuperId FROM Employee WHERE EmpId = 'e9';
 -- e9 | 1 | e8
UPDATE Employee SET DNumber = 2 WHERE EmpId = 'e9';

*/

-------------------------------------------------------------------------------
-- 16) The supervision relationship in Employee.SuperId must not be cyclic

-------------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION Employee_SuperId_NonCyclic()
RETURNS TRIGGER
LANGUAGE PLPGSQL AS $$
DECLARE
  oidfn Oid;
  total_rows integer;
BEGIN
  GET DIAGNOSTICS oidfn = PG_ROUTINE_OID;
  RAISE NOTICE 'Entering function %', oidfn::regproc;
  CREATE TEMPORARY TABLE Supervision (
    EmpId varchar(9),
    SuperId varchar(9),
    PRIMARY KEY (EmpId,SuperId) );
  INSERT INTO Supervision
  SELECT EmpId, SuperId
  FROM Employee
  WHERE SuperId IS NOT NULL;
  -- while previous operation affected some rows
  GET DIAGNOSTICS total_rows := ROW_COUNT;
  WHILE total_rows != 0 
  LOOP
    IF EXISTS ( 
      SELECT *
      FROM Supervision
      WHERE EmpId = SuperId ) THEN
      RAISE EXCEPTION 
        'The supervision relationship is cyclic'
        USING HINT = TG_OP || ' on table ' || TG_TABLE_NAME;
    END IF;
    INSERT INTO Supervision
    SELECT DISTINCT s1.EmpId, s2.SuperId
    FROM Supervision s1, Supervision s2
    WHERE s1.SuperId = s2.EmpId AND NOT EXISTS (
      SELECT *
      FROM Supervision s3
      WHERE s3.EmpId = s1.EmpId AND s3.SuperId = s2.SuperId );
    GET DIAGNOSTICS total_rows := ROW_COUNT;
  END LOOP;
END; $$;

CREATE OR REPLACE TRIGGER Employee_SuperId_NonCyclic_ins
AFTER INSERT ON Employee
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_NonCyclic();

CREATE OR REPLACE TRIGGER Employee_SuperId_NonCyclic_upd
AFTER UPDATE ON Employee
FOR EACH STATEMENT
EXECUTE PROCEDURE Employee_SuperId_NonCyclic();

/*
INSERT INTO Employee VALUES
  ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
  '450 Stone, Houston, TX', 'F', 23000, 'e10', 3, '01-01-1980');
-- ERROR:  The supervision relationship is cyclic

BEGIN TRANSACTION;
  ALTER TABLE Employee DISABLE TRIGGER ALL;
  INSERT INTO Employee VALUES
    ('e10', 'Elizabeth', 'A', 'Tudor', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e11', 3, '01-01-1980');
  INSERT INTO Employee VALUES
    ('e11', 'James', 'A', 'Hepburn', '10-11-1960',
    '450 Stone, Houston, TX', 'F', 23000, 'e10', 3, '01-01-1980');
  ALTER TABLE Employee ENABLE TRIGGER ALL;
END TRANSACTION;

*/

-------------------------------------------------------------------------------
